<?php

namespace App\Console\Commands;

use App\Models\League;
use App\Models\Team;
use App\Models\Player;
use App\Services\FootballApiService;
use Illuminate\Console\Command;

class FetchFootballData extends Command
{
    protected $signature = 'football:fetch {league_id?} {season?} {--fixtures : Fetch only today fixtures} {--date= : Fetch fixtures for specific date (YYYY-MM-DD)} {--detail= : Fetch details for specific fixture ID} {--live : Fetch only live matches} {--teams : Fetch teams for league and season} {--standings : Fetch standings for league and season} {--players : Fetch players for league and season}';

    protected $description = 'Fetch and store football data from API';

    public function handle(FootballApiService $api)
    {
        if ($this->option('fixtures') || $this->option('date')) {
            $this->fetchFixtures($api, $this->option('date'));

            return;
        }

        if ($this->option('live')) {
            $this->fetchLiveMatches($api);

            return;
        }

        if ($this->option('teams')) {
            $leagueId = $this->argument('league_id');
            $season = $this->argument('season');
            if (! $leagueId || ! $season) {
                $this->error('League ID and Season are required for --teams');

                return;
            }
            $this->fetchTeams($api, $leagueId, $season);

            return;
        }

        if ($this->option('standings')) {
            $leagueId = $this->argument('league_id');
            $season = $this->argument('season');
            if (! $leagueId || ! $season) {
                $this->error('League ID and Season are required for --standings');

                return;
            }
            $this->fetchStandings($api, $leagueId, $season);

            return;
        }

        if ($this->option('players')) {
            $leagueId = $this->argument('league_id');
            $season = $this->argument('season');
            if (! $leagueId || ! $season) {
                $this->error('League ID and Season are required for --players');

                return;
            }
            $this->fetchPlayers($api, $leagueId, $season);

            return;
        }

        if ($this->option('detail')) {
            $this->fetchFixtureDetail($api, $this->option('detail'));

            return;
        }

        $leagueId = $this->argument('league_id');
        $season = $this->argument('season');

        if ($leagueId && $season) {
            $this->fetchLeagueSeasonFixtures($api, $leagueId, $season);

            return;
        }

        $leagueId = $leagueId ?? 39;
        // ... rest of the code for teams and players
    }

    protected function fetchLeagueSeasonFixtures($api, $leagueId, $season)
    {
        $this->info("Fetching fixtures for league $leagueId, season $season...");
        $fixtures = $api->getLeagueFixtures($leagueId, $season);
        $this->saveFixtures($fixtures);
    }

    protected function fetchFixtures($api, $date = null)
    {
        $date = $date ?? now()->format('Y-m-d');
        $this->info("Fetching fixtures for $date...");
        $fixtures = $api->getFixtures($date);
        $this->saveFixtures($fixtures);
    }

    protected function saveFixtures($fixtures)
    {
        foreach ($fixtures as $data) {
            $league = League::firstOrCreate(
                ['api_league_id' => $data['league']['id']],
                [
                    'name_en' => $data['league']['name'],
                    'is_enabled' => true,
                ]
            );

            // تحميل شعارات الفرق محلياً
            $homeLogo = $this->downloadLogo($data['teams']['home']['name'], $data['teams']['home']['logo']);
            $awayLogo = $this->downloadLogo($data['teams']['away']['name'], $data['teams']['away']['logo']);

            $matchData = [
                'league_id' => $league->id,
                'home_team_name_en' => $data['teams']['home']['name'],
                'home_team_logo' => $homeLogo,
                'away_team_name_en' => $data['teams']['away']['name'],
                'away_team_logo' => $awayLogo,
                'home_score' => $data['goals']['home'],
                'away_score' => $data['goals']['away'],
                'status' => $data['fixture']['status']['short'] == 'LIVE' || is_numeric($data['fixture']['status']['short']) ? $data['fixture']['status']['elapsed'] : $data['fixture']['status']['short'],
                'match_date' => \Carbon\Carbon::parse($data['fixture']['date']),
                'venue' => $data['fixture']['venue']['name'] ?? null,
                'city' => $data['fixture']['venue']['city'] ?? null,
                'raw_json' => $data,
            ];

            if ($data['fixture']['status']['short'] == 'FT') {
                $existingMatch = \App\Models\FootballMatch::where('api_fixture_id', $data['fixture']['id'])->first();
                if ($existingMatch && ! $existingMatch->finished_at) {
                    $matchData['finished_at'] = now();
                    $matchData['live_visible_until'] = now()->addHours(2);
                } elseif (! $existingMatch) {
                    // إذا كانت المباراة منتهية أصلاً عند أول سحب
                    $matchData['finished_at'] = \Carbon\Carbon::parse($data['fixture']['date'])->addMinutes(110);
                    $matchData['live_visible_until'] = \Carbon\Carbon::parse($data['fixture']['date'])->addMinutes(110);
                }
            }

            \App\Models\FootballMatch::updateOrCreate(
                ['api_fixture_id' => $data['fixture']['id']],
                $matchData
            );
        }
        $this->info('Fixtures updated!');
    }

    protected function fetchLiveMatches($api)
    {
        $this->info('Fetching live matches...');
        $fixtures = $api->getLiveFixtures();

        if (empty($fixtures)) {
            $this->info('No live matches found.');

            return;
        }

        foreach ($fixtures as $data) {
            $league = League::where('api_league_id', $data['league']['id'])->first();
            if (! $league) {
                continue;
            }

            $matchData = [
                'home_score' => $data['goals']['home'],
                'away_score' => $data['goals']['away'],
                'status' => $data['fixture']['status']['short'] == 'LIVE' || is_numeric($data['fixture']['status']['short']) ? $data['fixture']['status']['elapsed'] : $data['fixture']['status']['short'],
                'raw_json' => $data,
            ];

            if ($data['fixture']['status']['short'] == 'FT') {
                $existingMatch = \App\Models\FootballMatch::where('api_fixture_id', $data['fixture']['id'])->first();
                if ($existingMatch && ! $existingMatch->finished_at) {
                    $matchData['finished_at'] = now();
                    $matchData['live_visible_until'] = now()->addHours(2);
                }
            }

            \App\Models\FootballMatch::updateOrCreate(
                ['api_fixture_id' => $data['fixture']['id']],
                $matchData
            );
        }
        $this->info('Live matches updated!');
    }

    protected function fetchFixtureDetail($api, $fixtureId)
    {
        $this->info("Fetching details for fixture $fixtureId...");
        $data = $api->getFixtureDetails($fixtureId);

        if (! $data) {
            $this->error("No data found for fixture $fixtureId");

            return;
        }

        \App\Models\FootballMatch::updateOrCreate(
            ['api_fixture_id' => $data['fixture']['id']],
            [
                'home_score' => $data['goals']['home'],
                'away_score' => $data['goals']['away'],
                'status' => $data['fixture']['status']['short'] == 'LIVE' || is_numeric($data['fixture']['status']['short']) ? $data['fixture']['status']['elapsed'] : $data['fixture']['status']['short'],
                'raw_json' => $data,
            ]
        );
        $this->info('Fixture detail updated!');
    }

    protected function fetchTeams($api, $leagueId, $season)
    {
        $this->info("Fetching teams for league $leagueId, season $season...");
        $response = $api->getTeams($leagueId, $season);

        if (empty($response)) {
            $this->error("No teams found for league $leagueId, season $season");

            return;
        }

        $league = League::where('api_league_id', $leagueId)->first();
        if (! $league) {
            $this->error("League with API ID $leagueId not found in database.");

            return;
        }

        foreach ($response as $data) {
            $teamData = $data['team'];
            $venueData = $data['venue'];

            $logoPath = $this->downloadLogo($teamData['name'], $teamData['logo']);

            Team::updateOrCreate(
                ['api_team_id' => $teamData['id']],
                [
                    'league_id' => $league->id,
                    'name_en' => $teamData['name'],
                    'logo' => $logoPath,
                    'venue_name' => $venueData['name'],
                    'venue_city' => $venueData['city'],
                    'venue_capacity' => $venueData['capacity'],
                    'raw_json' => $data,
                ]
            );
        }

        $this->info(count($response).' teams updated for '.$league->name_en);
    }

    protected function fetchStandings($api, $leagueId, $season)
    {
        $this->info("Fetching standings for league $leagueId, season $season...");
        $standings = $api->getStandings($leagueId, $season);

        if (empty($standings)) {
            $this->error("No standings found for league $leagueId, season $season");

            return;
        }

        $league = League::where('api_league_id', $leagueId)->first();
        if (! $league) {
            $this->error("League with API ID $leagueId not found in database.");

            return;
        }

        $league->update(['standings_json' => $standings]);
        $this->info('Standings updated for '.$league->name_en);
    }

    protected function fetchPlayers($api, $leagueId, $season)
    {
        $this->info("Fetching players for league $leagueId, season $season...");
        $page = 1;
        $totalPages = 1;

        do {
            $this->info("Fetching page $page...");
            $response = $api->getPlayers($leagueId, $season, $page);

            if (empty($response['response'])) {
                break;
            }

            $totalPages = $response['paging']['total'] ?? 1;

            foreach ($response['response'] as $data) {
                $playerData = $data['player'];
                $statistics = $data['statistics'][0] ?? null;

                if (! $statistics) {
                    continue;
                }

                $team = Team::where('api_team_id', $statistics['team']['id'])->first();
                if (! $team) {
                    continue;
                }

                $photoPath = $this->downloadPlayerPhoto($playerData['name'], $playerData['photo']);

                Player::updateOrCreate(
                    ['api_player_id' => $playerData['id']],
                    [
                        'team_id' => $team->id,
                        'name_en' => $playerData['name'],
                        'position' => $statistics['games']['position'],
                        'photo' => $photoPath,
                        'stats_json' => $data['statistics'],
                    ]
                );
            }

            $page++;
        } while ($page <= $totalPages && $page <= 5); // تحديد أول 5 صفحات لتجنب استهلاك الـ API

        $this->info('Players updated!');
    }

    protected function downloadPlayerPhoto($name, $url)
    {
        if (! $url) {
            return null;
        }
        try {
            $filename = 'players/'.\Illuminate\Support\Str::slug($name).'.png';
            if (! \Illuminate\Support\Facades\Storage::disk('public')->exists($filename)) {
                $contents = @file_get_contents($url);
                if ($contents) {
                    \Illuminate\Support\Facades\Storage::disk('public')->put($filename, $contents);
                } else {
                    return $url;
                }
            }

            return $filename;
        } catch (\Exception $e) {
            return $url;
        }
    }

    protected function downloadLogo($name, $url)
    {
        if (! $url) {
            return null;
        }
        try {
            $filename = 'teams/'.\Illuminate\Support\Str::slug($name).'.png';
            if (! \Illuminate\Support\Facades\Storage::disk('public')->exists($filename)) {
                $contents = @file_get_contents($url);
                if ($contents) {
                    \Illuminate\Support\Facades\Storage::disk('public')->put($filename, $contents);
                } else {
                    return $url; // العودة للرابط الأصلي في حال الفشل
                }
            }

            return $filename;
        } catch (\Exception $e) {
            return $url;
        }
    }
}
