<?php

namespace App\Http\Controllers\Api\v1;

use App\Http\Controllers\Controller;
use App\Models\Favorite;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class FavoriteController extends Controller
{
    public function index(Request $request)
    {
        $favorites = $request->user()->favorites()->with('item')->get();

        return response()->json([
            'status' => 'success',
            'data' => $favorites,
        ]);
    }

    public function toggle(Request $request)
    {
        $request->validate([
            'item_type' => ['required', Rule::in(['league', 'team', 'player'])],
            'item_id' => 'required|integer',
        ]);

        $userId = $request->user()->id;
        $itemType = $request->item_type;
        $itemId = $request->item_id;

        // Map short names to actual model classes if using morphTo
        $map = [
            'league' => \App\Models\League::class,
            'team' => \App\Models\Team::class,
            'player' => \App\Models\Player::class,
        ];

        $modelClass = $map[$itemType];

        // Check if item exists
        if (! $modelClass::where('id', $itemId)->exists()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Item not found',
            ], 404);
        }

        $favorite = Favorite::where('user_id', $userId)
            ->where('item_type', $modelClass)
            ->where('item_id', $itemId)
            ->first();

        if ($favorite) {
            $favorite->delete();
            $status = 'removed';
        } else {
            Favorite::create([
                'user_id' => $userId,
                'item_type' => $modelClass,
                'item_id' => $itemId,
            ]);
            $status = 'added';
        }

        return response()->json([
            'status' => 'success',
            'message' => "Favorite {$status} successfully",
            'is_favorite' => ($status === 'added'),
        ]);
    }
}
