<?php

namespace App\Jobs;

use App\Models\FootballMatch;
use App\Models\League;
use App\Services\FootballApiService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class SyncLeagueResultsJob implements ShouldQueue
{
    use Queueable;

    protected int $leagueId;

    protected string $date;

    /**
     * Create a new job instance.
     */
    public function __construct(int $leagueId, string $date)
    {
        $this->leagueId = $leagueId;
        $this->date = $date;
    }

    /**
     * Execute the job.
     */
    public function handle(FootballApiService $apiService): void
    {
        $league = League::where('api_league_id', $this->leagueId)->first();

        if (! $league || ! $league->is_enabled) {
            return;
        }

        $dateObj = strtotime($this->date);
        $year = (int) date('Y', $dateObj);
        $month = (int) date('m', $dateObj);

        // Football seasons usually start in August. If date is Jan-June, it's likely part of the previous year's season ID
        $season = ($month <= 6) ? $year - 1 : $year;

        $fixtures = $apiService->getFixtures([
            'league' => $this->leagueId,
            'date' => $this->date,
            'season' => $season,
        ]);

        // Fallback to current year if previous year season didn't work (for leagues that follow calendar year)
        if (empty($fixtures) && $season !== $year) {
            $fixtures = $apiService->getFixtures([
                'league' => $this->leagueId,
                'date' => $this->date,
                'season' => $year,
            ]);
        }

        if (empty($fixtures)) {
            return;
        }

        foreach ($fixtures as $fixture) {
            $status = $fixture['fixture']['status']['short'] == 'LIVE' || is_numeric($fixture['fixture']['status']['short'])
                ? $fixture['fixture']['status']['elapsed']
                : $fixture['fixture']['status']['short'];

            // Store all matches
            $matchData = [
                'league_id' => $league->id,
                'match_date' => $fixture['fixture']['date'],
                'status' => $status,
                'home_team_name_en' => $fixture['teams']['home']['name'],
                'away_team_name_en' => $fixture['teams']['away']['name'],
                'home_score' => $fixture['goals']['home'],
                'away_score' => $fixture['goals']['away'],
                'venue' => $fixture['fixture']['venue']['name'],
                'city' => $fixture['fixture']['venue']['city'],
                'raw_json' => $fixture,
            ];

            if ($fixture['fixture']['status']['short'] == 'FT') {
                $existingMatch = FootballMatch::where('api_fixture_id', $fixture['fixture']['id'])->first();
                if ($existingMatch && ! $existingMatch->finished_at) {
                    $matchData['finished_at'] = now();
                    $matchData['live_visible_until'] = now()->addHours(2);
                } elseif (! $existingMatch) {
                    $matchData['finished_at'] = \Carbon\Carbon::parse($fixture['fixture']['date'])->addMinutes(110);
                    $matchData['live_visible_until'] = \Carbon\Carbon::parse($fixture['fixture']['date'])->addMinutes(110);
                }
            }

            FootballMatch::updateOrCreate(
                ['api_fixture_id' => $fixture['fixture']['id']],
                $matchData
            );
        }

        // Clear cache for this league and date
        Cache::forget("results:{$league->id}:{$this->date}");

        Log::info("Synced fixtures for league {$this->leagueId} on {$this->date}");
    }
}
